<?php

namespace App\Filament\Resources\GaleriaResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;

class ImagensRelationManager extends RelationManager
{
    protected static string $relationship = 'imagens'; // hasMany em Galeria

    protected static ?string $title = 'Imagens';

    public function form(Form $form): Form
    {
        // (mantém o form de 1 a 1, caso queira continuar permitindo criar individualmente)
        return $form->schema([
            Forms\Components\TextInput::make('nome')->label('Legenda')->maxLength(150),
            Forms\Components\FileUpload::make('link')
                ->label('Imagem')
                ->image()
                ->disk('public')
                ->directory(fn () => 'galerias/' . ($this->getOwnerRecord()->id ?? 'temp'))
                ->visibility('public')
                ->preserveFilenames()
                ->downloadable()
                ->required(),
        ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('link')->label('Imagem')->disk('public')->square(),
                Tables\Columns\TextColumn::make('nome')->label('Legenda')->wrap(),
                Tables\Columns\TextColumn::make('created_at')->label('Enviada em')->dateTime('d/m/Y H:i'),
            ])
            ->headerActions([
                // AÇÃO DE UPLOAD MÚLTIPLO
                Tables\Actions\Action::make('uploadMultiplas')
                    ->label('Enviar várias imagens')
                    ->icon('heroicon-o-photo')
                    ->form([
                        Forms\Components\FileUpload::make('imagens')
                            ->label('Selecione as imagens')
                            ->image()
                            ->multiple()               // << permite várias
                            ->minFiles(1)
                            ->maxFiles(50)             // ajuste se quiser
                            ->disk('public')
                            ->directory(fn () => 'galerias/' . ($this->getOwnerRecord()->id ?? 'temp'))
                            ->visibility('public')
                            ->preserveFilenames()
                            ->downloadable(),
                    ])
                    ->action(function (array $data): void {
                        $galeria = $this->getOwnerRecord();
                        foreach (($data['imagens'] ?? []) as $path) {
                            $nome = pathinfo($path, PATHINFO_FILENAME);
                            $galeria->imagens()->create([
                                'nome'       => $nome,
                                'link'       => $path,
                                'noticias_id'=> null,
                            ]);
                        }
                    })
                    ->successNotificationTitle('Imagens enviadas com sucesso!'),

                Tables\Actions\CreateAction::make()->label('Adicionar uma imagem'), // opcional manter
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }
}
